"""Minimal Marketplace API client for reseller integrations.

Usage example::

    from marketplace_client import MarketplaceClient

    client = MarketplaceClient(
        base_url="https://api.asr.example/api/v1",
        api_key="your-api-key",
    )
    catalog = client.list_catalog(service="telegram", country="RU")
    order = client.fast_purchase(service_code="telegram", country="RU")
    client.close()

Dependencies: install ``httpx`` (``pip install httpx``).
"""

from __future__ import annotations

from dataclasses import dataclass
from typing import Any, Dict, Iterable, Optional

import httpx

DEFAULT_TIMEOUT = 10.0


@dataclass(slots=True)
class FastPurchaseRequest:
    service_code: str
    country: str
    operator: Optional[str] = None
    price_limit: Optional[float] = None
    currency: str = "USD"
    callback_url: Optional[str] = None
    client_reference: Optional[str] = None
    idempotency_key: Optional[str] = None
    extras: Optional[Dict[str, Any]] = None
    rental_tier: Optional[str] = None
    rental_price: Optional[float] = None
    provider_id: Optional[str] = None


class MarketplaceClientError(Exception):
    """Raised when the API returns a non-successful status."""

    def __init__(self, status_code: int, payload: Any):
        self.status_code = status_code
        self.payload = payload
        message = f"API request failed with status {status_code}: {payload!r}"
        super().__init__(message)


class MarketplaceClient:
    """Minimal client for the marketplace fast-purchase API."""

    def __init__(
        self,
        base_url: str,
        api_key: str,
        *,
        timeout: float = DEFAULT_TIMEOUT,
        transport: httpx.BaseTransport | None = None,
    ) -> None:
        headers = {
            "X-API-Key": api_key,
            "User-Agent": "asr-marketplace-python/1.0",
        }
        self._client = httpx.Client(
            base_url=base_url.rstrip("/"),
            timeout=timeout,
            headers=headers,
            transport=transport,
        )

    def close(self) -> None:
        """Close the underlying HTTP client."""

        self._client.close()

    def __enter__(self) -> "MarketplaceClient":
        return self

    def __exit__(self, *exc_info: object) -> None:
        self.close()

    def list_catalog(
        self,
        *,
        service: Optional[str] = None,
        country: Optional[str] = None,
        limit: Optional[int] = None,
        offset: Optional[int] = None,
    ) -> list[dict[str, Any]]:
        """Fetch catalog entries filtered by service/country."""

        params = self._build_params(
            (
                ("service", service),
                ("country", country),
                ("limit", limit),
                ("offset", offset),
            )
        )
        response = self._client.get("/marketplace/catalog", params=params)
        return self._json_or_error(response)

    def list_availability(
        self,
        *,
        service: Optional[str] = None,
        country: Optional[str] = None,
        operator: Optional[str] = None,
        limit: Optional[int] = None,
        cursor: Optional[str] = None,
    ) -> dict[str, Any]:
        """List current availability for a given service/country."""

        params = self._build_params(
            (
                ("service", service),
                ("country", country),
                ("operator", operator),
                ("limit", limit),
                ("cursor", cursor),
            )
        )
        response = self._client.get("/marketplace/availability", params=params)
        return self._json_or_error(response)

    def fast_purchase(self, request: FastPurchaseRequest | None = None, /, **kwargs: Any) -> dict[str, Any]:
        """Create a fast-purchase reservation.

        You can either pass a :class:`FastPurchaseRequest` instance or keyword
        arguments matching its fields.
        """

        if request is None:
            request = FastPurchaseRequest(**kwargs)
        payload = {
            "service_code": request.service_code,
            "country": request.country,
            "operator": request.operator,
            "price_limit": request.price_limit,
            "currency": request.currency,
            "callback_url": request.callback_url,
            "client_reference": request.client_reference,
            "idempotency_key": request.idempotency_key,
            "extras": request.extras,
            "rental_tier": request.rental_tier,
            "rental_price": request.rental_price,
            "provider_id": request.provider_id,
        }
        response = self._client.post("/marketplace/fast-purchase/api", json=payload)
        return self._json_or_error(response)

    def _json_or_error(self, response: httpx.Response) -> Any:
        if response.is_success:
            return response.json()
        try:
            payload: Any = response.json()
        except ValueError:
            payload = response.text
        raise MarketplaceClientError(response.status_code, payload)

    @staticmethod
    def _build_params(pairs: Iterable[tuple[str, Any]]) -> dict[str, Any]:
        params: dict[str, Any] = {}
        for key, value in pairs:
            if value is None:
                continue
            params[key] = value
        return params


__all__ = [
    "FastPurchaseRequest",
    "MarketplaceClient",
    "MarketplaceClientError",
]
