// Lightweight Marketplace API client for Node.js integrators.
// Requires Node 18+ (built-in fetch) or polyfill providing global fetch.

const DEFAULT_TIMEOUT = 10_000;

class MarketplaceClientError extends Error {
  constructor(status, payload) {
    const message = `API request failed with status ${status}: ${JSON.stringify(payload)}`;
    super(message);
    this.name = "MarketplaceClientError";
    this.status = status;
    this.payload = payload;
  }
}

class MarketplaceClient {
  constructor({ baseUrl, apiKey, timeout = DEFAULT_TIMEOUT } = {}) {
    if (!baseUrl) {
      throw new Error("baseUrl is required");
    }
    if (!apiKey) {
      throw new Error("apiKey is required");
    }
    this.baseUrl = baseUrl.replace(/\/$/, "");
    this.apiKey = apiKey;
    this.timeout = timeout;
    this.defaultHeaders = {
      "X-API-Key": this.apiKey,
      "User-Agent": "asr-marketplace-node/1.0",
    };
  }

  async listCatalog({ service, country, limit, offset } = {}) {
    const params = this.#encodeParams({ service, country, limit, offset });
    return this.#request(`/marketplace/catalog${params}`);
  }

  async listAvailability({ service, country, operator, limit, cursor } = {}) {
    const params = this.#encodeParams({ service, country, operator, limit, cursor });
    return this.#request(`/marketplace/availability${params}`);
  }

  async fastPurchase(payload) {
    if (!payload || !payload.service_code || !payload.country) {
      throw new Error("fastPurchase payload must include service_code and country");
    }
    return this.#request(`/marketplace/fast-purchase/api`, {
      method: "POST",
      headers: { "Content-Type": "application/json" },
      body: JSON.stringify(payload),
    });
  }

  async #request(path, options = {}) {
    const controller = new AbortController();
    const timer = setTimeout(() => controller.abort(), this.timeout);
    try {
      const response = await fetch(`${this.baseUrl}${path}`, {
        ...options,
        headers: {
          ...this.defaultHeaders,
          ...(options.headers || {}),
        },
        signal: controller.signal,
      });
      const contentType = response.headers.get("content-type") || "";
      const isJson = contentType.includes("application/json");
      const payload = isJson ? await response.json() : await response.text();
      if (!response.ok) {
        throw new MarketplaceClientError(response.status, payload);
      }
      return payload;
    } catch (error) {
      if (error.name === "AbortError") {
        throw new Error(`Request timed out after ${this.timeout} ms`);
      }
      throw error;
    } finally {
      clearTimeout(timer);
    }
  }

  #encodeParams(values) {
    const entries = Object.entries(values || {}).filter(([, value]) => value !== undefined && value !== null);
    if (entries.length === 0) {
      return "";
    }
    const search = new URLSearchParams();
    for (const [key, value] of entries) {
      search.set(key, String(value));
    }
    return `?${search.toString()}`;
  }
}

module.exports = {
  MarketplaceClient,
  MarketplaceClientError,
};
